package kz.dogovor24.module.services;

import java.io.FileInputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.KeyStore;
import java.security.MessageDigest;
import java.security.PrivateKey;
import java.security.Security;
import java.security.cert.CertStore;
import java.security.cert.Certificate;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.security.auth.x500.X500Principal;

import org.json.JSONArray;
import org.json.JSONObject;

import kz.dogovor24.module.model.EsedoConsts;
import kz.dogovor24.module.model.ResponseJson;
import kz.dogovor24.module.model.ResultWrapper;
import kz.dogovor24.module.model.StorageInfo;
import kz.gov.pki.kalkan.asn1.ASN1EncodableVector;
import kz.gov.pki.kalkan.asn1.DERGeneralizedTime;
import kz.gov.pki.kalkan.asn1.DERObjectIdentifier;
import kz.gov.pki.kalkan.asn1.DEROctetString;
import kz.gov.pki.kalkan.asn1.DERPrintableString;
import kz.gov.pki.kalkan.asn1.DERSet;
import kz.gov.pki.kalkan.asn1.DERUTF8String;
import kz.gov.pki.kalkan.asn1.cms.Attribute;
import kz.gov.pki.kalkan.asn1.cms.AttributeTable;
import kz.gov.pki.kalkan.asn1.cryptopro.CryptoProObjectIdentifiers;
import kz.gov.pki.kalkan.asn1.knca.KNCAObjectIdentifiers;
import kz.gov.pki.kalkan.asn1.pkcs.PKCSObjectIdentifiers;
import kz.gov.pki.kalkan.asn1.x509.X509Name;
import kz.gov.pki.kalkan.jce.provider.KalkanProvider;
import kz.gov.pki.kalkan.jce.provider.cms.CMSProcessableByteArray;
import kz.gov.pki.kalkan.jce.provider.cms.CMSSignedData;
import kz.gov.pki.kalkan.jce.provider.cms.CMSSignedDataGenerator;
import kz.gov.pki.kalkan.jce.provider.cms.CMSSignedGenerator;
import kz.gov.pki.kalkan.ocsp.CertificateID;
import kz.gov.pki.kalkan.ocsp.OCSPReq;
import kz.gov.pki.kalkan.ocsp.OCSPReqGenerator;
import kz.gov.pki.kalkan.ocsp.OCSPResp;
import kz.gov.pki.kalkan.util.encoders.Base64;
import kz.gov.pki.provider.utils.KeyStoreUtil;

public class EsedoSignService {

	final String OCSP_URL;

	KeyStore _keyStore = null;

	String _alias = null;

	Map<X500Principal, X509Certificate> _caCertsMap;

	public EsedoSignService() {
		this.OCSP_URL = (System.getProperty("knca.ocspresponderURL") == null) ? "http://ocsp.pki.gov.kz"
				: System.getProperty("knca.ocspresponderURL");
	}



	public String process(String jsonString,StorageInfo storageInfo )throws Exception {


		String resultObjectStr = null;
		ResponseJson<String> responseForJS = new ResponseJson<>("500");
		String uuid = "";
		try {
			//Provider provider = kalkanProvider();
			responseForJS.setResponseObject(null);
			responseForJS.setCode("500");
			JSONObject jsonObject = new JSONObject(jsonString);
			uuid = jsonObject.optString("uuid");
			JSONArray args = jsonObject.getJSONArray("args");
			String dataToSign = args.get(0).toString();
			String signFileNames = args.get(1).toString();
			List<X509Certificate> caCerts = KeyStoreUtil.getDefaultCACerts();
			this._caCertsMap = new HashMap<>();
			for (X509Certificate x509Certificate : caCerts) {
				this._caCertsMap.put(x509Certificate.getSubjectX500Principal(), x509Certificate);
			}
			this._keyStore = KeyStore.getInstance("PKCS12", "KALKAN");
			InputStream inputStream = new FileInputStream(storageInfo.getContainer());
			try {
				this._keyStore.load(inputStream, storageInfo.getPassword().toCharArray());
			} catch (Exception e) {
			}
			Enumeration<String> als = this._keyStore.aliases();
			while (als.hasMoreElements()) {
				this._alias = als.nextElement();
			}
			X509Certificate cert = (X509Certificate) this._keyStore.getCertificate(this._alias);
			byte[] ocspResponse = getOcspTicket(cert);
			if (ocspResponse == null) {
			}
			/*cms signing*/
			try {
				ResultWrapper result = signCMS(Base64.decode(dataToSign), storageInfo.getPassword(), signFileNames,
						ocspResponse);
				if(result == null) {
					responseForJS.setCode("500");
					responseForJS.setMessage("Необходимо использовать ключ для подписи!");
					JSONObject resultObject = (JSONObject) JSONObject.wrap(responseForJS);
					resultObjectStr = resultObject.put("uuid", uuid).toString();
					return resultObjectStr;
				}
				responseForJS.setCode("200");
				responseForJS.setMessage(result.getMsg());
				responseForJS.setResponseObject(result.getResult());
				JSONObject resultObject = (JSONObject) JSONObject.wrap(responseForJS);
				resultObjectStr = resultObject.put("uuid", uuid).toString();
			}
			catch(Exception ex) {
				responseForJS.setCode("500");
				responseForJS.setMessage(ex.toString());
				JSONObject resultObject = (JSONObject) JSONObject.wrap(responseForJS);
				resultObjectStr = resultObject.put("uuid", uuid).toString();
			}
			
		} catch (Exception e) {
			responseForJS.setCode("500");
			responseForJS.setMessage(e.toString());
			JSONObject resultObject = (JSONObject) JSONObject.wrap(responseForJS);
			resultObjectStr = resultObject.put("uuid", uuid).toString();
		}
		return resultObjectStr;
	}

	private ResultWrapper signCMS(byte[] dataToSign, String password, String signFileNames, byte[] ocspResponse) throws Exception {
		ResultWrapper rw = null;
		try {
			PrivateKey privateKey = (PrivateKey) this._keyStore.getKey(this._alias, password.toCharArray());
			X509Certificate x509Certificate = (X509Certificate) this._keyStore.getCertificate(this._alias);
			MessageDigest messageDigest = MessageDigest.getInstance("GOST3411", "KALKAN");
			messageDigest.reset();	
			 if (x509Certificate.getSigAlgOID()
						.equals(KNCAObjectIdentifiers.gost3411_2015_with_gost3410_2015_512.getId())) {
				 	messageDigest = MessageDigest.getInstance("GOST3411-2015-512", "KALKAN");
					messageDigest.reset();	 
			 }
			List<String> keyusage = x509Certificate.getExtendedKeyUsage();
			if (keyusage.contains("1.3.6.1.5.5.7.3.2")) {
				throw new Exception("Необходимо использовать ключ для подписи!");
			}
			ASN1EncodableVector attr = new ASN1EncodableVector();
			
			byte[] hash = messageDigest.digest(dataToSign);
			attr.add(new Attribute(new DERObjectIdentifier(EsedoConsts.OID_SIGN_HASH),
					new DERSet(new DEROctetString(hash))));
			Date currentdate = new Date();
			Attribute signTimeAttribute = new Attribute(new DERObjectIdentifier(EsedoConsts.OID_SIGN_DATE),
					new DERSet(new DERGeneralizedTime(currentdate)));
			attr.add(signTimeAttribute);
			if (signFileNames != null) {
				attr.add(new Attribute(new DERObjectIdentifier(EsedoConsts.OID_SIGNED_FILE_NAMES),
						new DERSet(new DERUTF8String(signFileNames))));
			}
			if (ocspResponse != null) {
				attr.add(new Attribute(new DERObjectIdentifier(EsedoConsts.OID_OCSP_RESPONSE),
						new DERSet(new DEROctetString(ocspResponse))));
			}
			attr.add(new Attribute(new DERObjectIdentifier(EsedoConsts.OID_DN_NAME),
					new DERSet(new X509Name(x509Certificate.getSubjectX500Principal().getName()))));
			attr.add(new Attribute(new DERObjectIdentifier(EsedoConsts.OID_CONTENT_TYPE),
					new DERSet(new DERObjectIdentifier(EsedoConsts.OID_SIGNED_CONTENT_TYPE))));
			String descStr = "ESEDO";
			byte[] descr = new byte[descStr.length() + 1];
			System.arraycopy(descStr.getBytes(), 0, descr, 0, descStr.length());
			attr.add(new Attribute(new DERObjectIdentifier(EsedoConsts.OID_SIGNING_DESCRIPTION),
					new DERSet(new DERPrintableString(descr))));
			AttributeTable attributeTable = new AttributeTable(attr);
			CertStore chainStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(
					Arrays.asList((Object[]) new X509Certificate[] { x509Certificate })), "KALKAN");
			CMSSignedDataGenerator generator = new CMSSignedDataGenerator();

			if (x509Certificate.getSigAlgOID().equals(PKCSObjectIdentifiers.sha1WithRSAEncryption.getId())) {
				generator.addSigner(privateKey, x509Certificate, CMSSignedGenerator.DIGEST_SHA1, attributeTable,
						null);
			} else if (x509Certificate.getSigAlgOID().equals(PKCSObjectIdentifiers.sha256WithRSAEncryption.getId())) {
				generator.addSigner(privateKey, x509Certificate, CMSSignedGenerator.DIGEST_SHA256, attributeTable,
						null);
			} else if (x509Certificate.getSigAlgOID()
					.equals(KNCAObjectIdentifiers.gost34311_95_with_gost34310_2004.getId())) {
				generator.addSigner(privateKey, x509Certificate, CMSSignedGenerator.DIGEST_GOST34311_95,
						attributeTable, null);
			} else if (x509Certificate.getSigAlgOID()
					.equals(CryptoProObjectIdentifiers.gostR3411_94_with_gostR34310_2004.getId())) {
				generator.addSigner(privateKey, x509Certificate, CMSSignedGenerator.DIGEST_GOST3411_GT,
						attributeTable, null);
				
			}
			else if (x509Certificate.getSigAlgOID()
					.equals(KNCAObjectIdentifiers.gost3411_2015_with_gost3410_2015_512.getId())) {
				generator.addSigner(privateKey, x509Certificate, CMSSignedDataGenerator.DIGEST_GOST3411_2015_512,
						attributeTable, null);
			}
			else {
				throw new Exception();
			}
			generator.addCertificatesAndCRLs(chainStore);
			CMSProcessableByteArray cMSProcessableByteArray = new CMSProcessableByteArray(dataToSign);
			CMSSignedData signedData = generator.generate(cMSProcessableByteArray, false, "KALKAN");
			byte[] signedDataEncoded = signedData.getEncoded();
			rw = new ResultWrapper();
			rw.setResult(new String(Base64.encode(signedDataEncoded)));
			//String encodedData = new String(Base64.encode(signedDataEncoded));

		} catch (Exception e) {
			rw = null;
		} finally {
			this._keyStore = null;
		}
		return rw;
	}

	private byte[] getOcspTicket(X509Certificate certificate) {
		byte[] ocspTicketArray;
		try {
			HttpURLConnection con;
			X509Certificate caCert = this._caCertsMap.get(certificate.getIssuerX500Principal());
			byte[] ocspReq = getOcspPackage(certificate.getSerialNumber(), caCert, CertificateID.HASH_SHA256);
			String b64Req = new String(Base64.encode(ocspReq));
			String getUrl = this.OCSP_URL + "/" + b64Req;
			if (getUrl.length() <= 2) {
				URL url = new URL(getUrl);
				con = (HttpURLConnection) url.openConnection();
			} else {
				URL url = new URL(this.OCSP_URL);
				con = (HttpURLConnection) url.openConnection();
				con.setDoOutput(true);
				con.setRequestMethod("POST");
				con.setRequestProperty("Content-Type", "application/ocsp-request");
				OutputStream os = con.getOutputStream();
				os.write(ocspReq);
				os.close();
			}
			InputStream inputStream = con.getInputStream();
			OCSPResp response = new OCSPResp(inputStream);
			ocspTicketArray = response.getEncoded();
			con.disconnect();
		} catch (Exception e) {
			ocspTicketArray = null;
		}
		return ocspTicketArray;
	}

	private byte[] getOcspPackage(BigInteger serialNr, Certificate cacert, String hashAlg) throws Exception {
		OCSPReqGenerator gen = new OCSPReqGenerator();
		CertificateID certId = new CertificateID(hashAlg, (X509Certificate) cacert, serialNr, "KALKAN");
		gen.addRequest(certId);
		OCSPReq req = gen.generate();
		return req.getEncoded();
	}

	public KalkanProvider kalkanProvider() {
		KalkanProvider kalkanProvider = new KalkanProvider();
		Security.addProvider(kalkanProvider);
		return kalkanProvider;
	}

}
